package easik.states;

import javax.swing.JOptionPane;

import easik.Easik;
import easik.sketch.Sketch;
import easik.sketch.constraint.CommutativeDiagram;
import easik.sketch.constraint.Constraint;
import easik.sketch.path.SketchPath;


/** 
 * This state makes a commutative diagram. When pushed, it will then push another
 * state to get a path. Once the first path is received, it will push another request for
 * a path. Once two paths are accepted, it will create the constraint and then add it
 * to the sketch.
 *
 * @author Rob Fletcher 2005
 * @author Kevin Green 2006
 * @author Vera Ranieri 2006
 * @version 2006-08-15 Kevin Green
 */
public class AddCommutativeDiagramState	extends EasikState implements PathAcceptingState {
	
	/**
	 * The first path involved in this constraint
	 */
	private SketchPath _pathA;
	/**
	 * The second path involved in this constraint
	 */
	private SketchPath _pathB;

	/**
	 * Boolean to ascertain whether the selection of paths for this constraint has been finished
	 */
	private boolean _finished;
	
	/**
	 * Constructor for creating a new state to collect paths and make a diagram.
	 * 
	 * @param inSketch The sketch being worked with.
	 */
	public AddCommutativeDiagramState(Sketch inSketch) {
		_ourSketch = inSketch;
		_finished = false;
	}

	/**
	 * Called when a new path has been completed by a state above this on the
	 * stack. If it is null, then cancel, otherwise add it to the collection.
	 * When two paths are in the collection, finish (if they share a domain and codomain).
	 * 
	 * @param inPath The last path in, null if it was a cancellation
	 */
	public void passPath(SketchPath inPath) {
		_finished = Easik.getInstance().getStateManager().getFinished();
		//Check if cancelled
		if (inPath == null) {
			Easik.getInstance().getStateManager().popState();
			return;
		}
		
		//Check which path is being passed
		if(!_finished){
			
			_pathA = inPath;
			Easik.getInstance().getStateManager().pushState(new GetPathState(false, true));
			
		}
		else{
			_pathB = inPath;
			if(addDiagram()){
				JOptionPane.showMessageDialog(
					_ourSketch.getParent(),
					"Created a Commutative Diagram",
					"JEase",
					JOptionPane.INFORMATION_MESSAGE);
			}
			else{
				JOptionPane.showMessageDialog(_ourSketch.getParent(), 
						"Invalid path selection.\nCommutative Diagram not created.", 
						"Error", JOptionPane.ERROR_MESSAGE);
			}
			Easik.getInstance().getStateManager().popState();
		}
	}

	/**
	 * Add the diagram to the sketch
	 * 
	 * @return true if the diagram was successfully added to the sketch, false otherwise
	 */
	private boolean addDiagram() {
		if(CommutativeDiagram.isCommutativeDiagram(_pathA, _pathB)){
			Constraint newDiagram = new CommutativeDiagram(_pathA, _pathB);
			Easik.getInstance().getFrame().getSketch().addNewConstraint(newDiagram);
			Easik.getInstance().getFrame().getSketch().setDirty(true);
			return true;
		}
		return false;
	}

	/**
	 * When this state is pushed on, it sends a message in a popup and then pushes
	 * a path collection state.
	 */
	public void pushedOn() {
		Easik.getInstance().getFrame().getSketch().getSelectionListener().emptySelection();
		Easik.getInstance().getStateManager().resetFinished();
		JOptionPane.showMessageDialog(
			_ourSketch.getParent(),
			"Select the first path and press 'Next'.\nThen select the second path and click 'Finish'",
			"JEase",
			JOptionPane.INFORMATION_MESSAGE);

		Easik.getInstance().getFrame().setStateString("Commutative Diagram");
		Easik.getInstance().getFrame().getNextButton().setEnabled(true);
		Easik.getInstance().getFrame().getCancelButton().setEnabled(true);
		Easik.getInstance().getStateManager().pushState(new GetPathState(true,false));
	}

	/**
	 * Nothing to do when it gets popped off, as it always pops itself off when completed. 
	 */
	public void poppedOff() {
	}

	/**
	 * This state is called "New Commutative diagram"
	 * 
	 * @return The string literal described above.
	 */
	public String toString() {
		return "New Commutative Diagram";
	}

}
